﻿;------------------------------------------------------------------------------
; AccViewer.ahk
; by jethrow
; Updated by jeeswg (for AHK v2.0+)
; last updated 2023-04-19
;------------------------------------------------------------------------------

#Requires AutoHotkey v1.1.35+
#Include %A_ScriptDir%\JEEAhk1FC.ahk
#Include %A_ScriptDir%\JEEAhk1FCGui.ahk
;for AutoHotkey v1.1.35+:
;replace '#HotIf' with '#If'
;replace '&' with 'ByRef ' (in GetAccPath/GetAccLocation/Acc_ObjectFromPoint/Acc_Location)
;use backport libraries (e.g. JEEAhk1FC.ahk/JEEAhk1FCGui.ahk)

;classes:
class Cursor
{
	__New(id)
	{
		this.Handle := DllCall("user32\LoadCursor", "Ptr",0, "Ptr",id, "Ptr")
	}
	__Delete()
	{
		DllCall("user32\DestroyCursor", "Ptr",this.Handle)
	}
}
class Outline
{
	__New(color:="red")
	{
		this.Bars := []
		Loop 4
		{
			this.Bars.Push(Gui("-Caption +ToolWindow"))
			this.Bars[A_Index].BackColor := color
		}
		this._Visible := False
		this.SetColor(color)
		this.Top := this.Bars[1]
		this.Right := this.Bars[2]
		this.Bottom := this.Bars[3]
		this.Left := this.Bars[4]
	}
	Show(x1, y1, x2, y2, sides:="TRBL")
	{
		InStr(sides, "T") ? this.Bars[1].Show("NA X" x1-2 " Y" y1-2 " W" x2-x1+4 " H" 2) : this.Bars[1].Hide()
		InStr(sides, "R") ? this.Bars[2].Show("NA X" x2 " Y" y1 " W" 2 " H" y2-y1) : this.Bars[2].Hide()
		InStr(sides, "B") ? this.Bars[3].Show("NA X" x1-2 " Y" y2 " W" x2-x1+4 " H" 2) : this.Bars[3].Hide()
		InStr(sides, "L") ? this.Bars[4].Show("NA X" x1-2 " Y" y1 " W" 2 " H" y2-y1) : this.Bars[4].Hide()
		this._Visible := True
	}
	Hide()
	{
		Loop 4
			this.Bars[A_Index].Hide()
		this._Visible := False
	}
	SetAbove(Hwnd)
	{
		Above := DllCall("user32\GetWindow", "Ptr",Hwnd, "UInt",3, "Ptr")
		Loop 4 ;SWP_NOACTIVATE := 0x10 ;SWP_NOMOVE := 0x2 ;SWP_NOSIZE := 0x1
			DllCall("user32\SetWindowPos", "Ptr",this.Bars[A_Index].Hwnd, "Ptr",Above, "Int",0, "Int",0, "Int",0, "Int",0, "UInt",0x1 | 0x2 | 0x10)
	}
	SetTransparent(param)
	{
		Loop 4
			WinSetTransparent(param = 1 ? 0 : 255, "ahk_id " this.Bars[A_Index].Hwnd)
		this._Visible := !param
	}
	SetColor(color)
	{
		Loop 4
			this.Bars[A_Index].BackColor := color
		this._Color := color
	}
	Destroy()
	{
		Loop 4
			this.Bars[A_Index].Destroy()
	}
}

;initial code:
if 1
{
	;some global variables:
	WM_ACTIVATE := 0x06
	WM_KILLFOCUS := 0x08
	WM_LBUTTONDOWN := 0x201
	WM_LBUTTONUP := 0x202
	IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	ClassName := new := "Outline" ;note: give 'new' a dummy value so AHK v2 doesn't complain
	Border := IsV1 ? new %ClassName%() : %ClassName%()
	Stored := {CHwnd:"", Hwnd:"", Location:""}
	Acc := 0 ;will be an object
	ChildID := 0
	TVobj := 0 ;will be an object
	ShowingLess := ""
	LButton_Pressed := False
	WHwnd := 0
	_GetClassNN := {}
	CH := False

	;some GUI control variables:
	ShowStructure := ""
	HBar := ""
	VBar := ""
	WinCtrl := ""
	ClassText := ""
	AccLocationText := ""
	TView := 0
}
if 1
{
	DetectHiddenWindows(1)
	OnExit(IsV1?"OnExitCleanup":OnExitCleanup)
	OnMessage(0x200, IsV1?"WM_MOUSEMOVE":WM_MOUSEMOVE)
	;ComObjError(False)
	Hotkey("~LButton Up", "Off")
}
if 1
{
	oGuiMain := Gui("+AlwaysOnTop", "Accessible Info Viewer")
	oGuiMain.OnEvent("Close", IsV1?Func("GuiMainClose"):GuiMainClose)
	oBtn := oGuiMain.Add("Button", "x160 y8 w105 h20 vShowStructure", "Show Acc Structure")
	oBtn.OnEvent("Click", IsV1?Func("ShowStructureSub"):ShowStructureSub)
	if 1
	{
		oStc1 := oGuiMain.Add("Text", "x10 y3 w25 h26 Border ReadOnly")
		oStc1.OnEvent("Click", IsV1?Func("CrossHairSub"):CrossHairSub)
		CColor(oStc1.Hwnd, "White")
		oStc2 := oGuiMain.Add("Text", "x10 y3 w25 h4 Border")
		CColor(oStc2.Hwnd, "0046D5")
		oGuiMain.Add("Text", "x13 y17 w19 h1 Border vHBar")
		oGuiMain.Add("Text", "x22 y8 w1 h19 Border vVBar")
	}
	if 1
	{
		;note: appended '_' to some GUI variable names: Hwnd/Class)
		oGuiMain.SetFont("Bold")
		oGuiMain.Add("GroupBox", "x2 y32 w275 h130 vWinCtrl", "Window/Control Info")
		oGuiMain.SetFont("Norm")
		oGuiMain.Add("Text", "x7 y49 w42 h20 Right", "WinTitle:")
		oGuiMain.Add("Edit", "x51 y47 w221 h20 vWinTitle")
		oGuiMain.Add("Text", "x7 y71 w42 h20 Right", "Text:")
		oGuiMain.Add("Edit", "x51 y69 w221 h20 vText")
		oGuiMain.Add("Text", "x7 y93 w42 h20 Right", "Hwnd:")
		oGuiMain.Add("Edit", "x51 y91 w72 h20 vHwnd_") ;AHK v1: 'Hwnd' key would clash with 'Hwnd' property, so use 'Hwnd_'
		oGuiMain.Add("Text", "x126 y93 w51 h20 vClassText Right", "Class(NN):")
		oGuiMain.Add("Edit", "x178 y91 w94 h20 vClass_")
		oGuiMain.Add("Text", "x7 y115 w42 h20 Right", "Position:")
		oGuiMain.Add("Edit", "x51 y113 w72 h20 vPosition")
		oGuiMain.Add("Text", "x126 y115 w51 h20 Right", "Process:")
		oGuiMain.Add("Edit", "x178 y113 w94 h20 vProcess")
		oGuiMain.Add("Text", "x7 y137 w42 h20 Right", "Size:")
		oGuiMain.Add("Edit", "x51 y135 w72 h20 vSize")
		oGuiMain.Add("Text", "x126 y137 w51 h20 Right", "Proc ID:")
		oGuiMain.Add("Edit", "x178 y135 w94 h20 vProcID")
	}
	if 1
	{
		oGuiMain.SetFont("Bold")
		oGuiMain.Add("GroupBox", "x2 y165 w275 h240 vAcc", "Accessible Info")
		oGuiMain.SetFont("Norm")
		oGuiMain.Add("Text", "x7 y182 w42 h20 Right", "Name:")
		oGuiMain.Add("Edit", "x51 y180 w221 h20 vAccName")
		oGuiMain.Add("Text", "x7 y204 w42 h20 Right", "Value:")
		oGuiMain.Add("Edit", "x51 y202 w221 h20 vAccValue")
		oGuiMain.Add("Text", "x7 y226 w42 h20 Right", "Role:")
		oGuiMain.Add("Edit", "x51 y224 w72 h20 vAccRole")
		oGuiMain.Add("Text", "x126 y226 w55 h20 Right", "ChildCount:")
		oGuiMain.Add("Edit", "x182 y224 w90 h20 vAccChildCount")
		oGuiMain.Add("Text", "x7 y248 w42 h20 Right", "State:")
		oGuiMain.Add("Edit", "x51 y246 w72 h20 vAccState")
		oGuiMain.Add("Text", "x126 y248 w55 h20 Right", "Selection:")
		oGuiMain.Add("Edit", "x182 y246 w90 h20 vAccSelection")
		oGuiMain.Add("Text", "x7 y270 w42 h20 Right", "Action:")
		oGuiMain.Add("Edit", "x51 y268 w72 h20 vAccAction")
		oGuiMain.Add("Text", "x126 y270 w55 h20 Right", "Focus:")
		oGuiMain.Add("Edit", "x182 y268 w90 h20 vAccFocus")
		if 1
		{
			oGuiMain.Add("Text", "x7 y292 w55 h20 Right vAccLocationText", "Location:")
			oGuiMain.Add("Edit", "x65 y290 w207 h20 vAccLocation")
			oGuiMain.Add("Text", "x7 y314 w55 h20 Right", "Description:")
			oGuiMain.Add("Edit", "x65 y312 w207 h20 vAccDescription")
			oGuiMain.Add("Text", "x7 y336 w55 h20 Right", "Keyboard:")
			oGuiMain.Add("Edit", "x65 y334 w207 h20 vAccKeyboard")
			oGuiMain.Add("Text", "x7 y358 w55 h20 Right", "Help:")
			oGuiMain.Add("Edit", "x65 y356 w207 h20 vAccHelp")
			oGuiMain.Add("Text", "x7 y380 w55 h20 Right", "HelpTopic:")
			oGuiMain.Add("Edit", "x65 y378 w207 h20 vAccHelpTopic")
		}
	}
	if 1
	{
		oSB := oGuiMain.Add("StatusBar")
		oSB.OnEvent("Click", IsV1?Func("ShowMainGui"):ShowMainGui)
		oSB.SetParts(70, 150)
		oSB.SetText("`tshow more", 3)
	}
	if 1
	{
		oGuiAcc := Gui("+ToolWindow +AlwaysOnTop +Resize", "Acc Structure")
		oGuiAcc.OnEvent("Close", IsV1?Func("GuiAccClose"):GuiAccClose)
		oGuiAcc.OnEvent("Size", IsV1?Func("GuiAccSize"):GuiAccSize)
		oTV := oGuiAcc.Add("TreeView", "w200 h300 vTView R17")
		oTV.OnEvent("ItemSelect", IsV1?Func("TreeView_ItemSelect"):TreeView_ItemSelect)
		oTV.OnEvent("ItemExpand", IsV1?Func("TreeView_ItemExpand"):TreeView_ItemExpand)
		oGuiAcc.Show("Hide") ;note: resizes window to fit controls (window is already hidden) (appears to be unnecessary in AHK v2)
	}
	ShowMainGui()
	;WinRedraw("ahk_id " oGuiMain.Hwnd) ;[FIXME] necessary?
}
return ;end of the auto-execute section

ShowMainGui(oCtl:="", Part:=0, oParams*)
{
	global
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	if (Part = 1) || (Part = 2) ;which part of the status bar was clicked
	{
		WM_MOUSEMOVE()
		SB_Text1 := StatusBarGetText(1, "ahk_id " oGuiMain.Hwnd)
		SB_Text2 := StatusBarGetText(2, "ahk_id " oGuiMain.Hwnd)
		if (SB_Text2 != "")
		{
			;update: the path is put onto the clipboard as before, but now also a child ID if present:
			;update: a click on part 1 or part 2 of the status bar, gives the same output:
			SB_Text2 := SubStr(SB_Text2, 7)
			if RegExMatch(SB_Text1, "^Child Id: \d+$")
				SB_Text2 .= " c" RegExReplace(SB_Text1, "\D")
			ToolTip("clipboard = " A_Clipboard := SB_Text2)
			SetTimer(IsV1?"RemoveToolTip":RemoveToolTip, -2000)
		}
	}
	else
	{
		if ShowingLess
		{
			oSB.SetText("`tshow less", 3)
			;oGuiMain["Acc"].Move("x2 y165 w275 h240") ;[FIXME] necessary?
			oGuiMain["AccDescription"].Visible := True
			oGuiMain["AccLocation"].Visible := True
			oGuiMain["AccLocationText"].Visible := True
			if 1
			{
				height := 319
				while (height < 428)
				{
					height += 10
					oGuiMain.Show("w280 h" height)
					Sleep(20)
				}
			}
			oGuiMain.Show("w280 h428")
			ShowingLess := False
			oSB.Redraw() ;status bar sometimes needs redrawing after pressing 'show more'
		}
		else
		{
			if (ShowingLess != "")
			{
				height := 428
				while (height > 319)
				{
					height -= 10
					oGuiMain.Show("w280 h" height)
					Sleep(20)
				}
			}
			oGuiMain.Show("w280 h319")
			oGuiMain["AccDescription"].Visible := False
			oGuiMain["AccLocation"].Visible := False
			oGuiMain["AccLocationText"].Visible := False
			;oGuiMain["Acc"].Move("x2 y165 w275 h130") ;[FIXME] necessary?
			oSB.SetText("`tshow more", 3)
			ShowingLess := True
		}
		;WinRedraw("ahk_id " oGuiMain.Hwnd) ;[FIXME] necessary?
	}
}

;hotkeys:
#If !LButton_Pressed
^/::HotkeyLB0()
HotkeyLB0()
{
	global
	LButton_Pressed := True
	Stored.CHwnd := ""
	oTV.Enabled := False
	while LButton_Pressed
		GetAccInfo()
}
#If LButton_Pressed
^/::HotkeyLB1()
HotkeyLB1()
{
	global
	LButton_Pressed := False
	Sleep(-1)
	oGuiMain["WinCtrl"].Text := (DllCall("user32\GetParent", "Ptr",Acc_WindowFromObject(Acc), "Ptr") ? "Control":"Window") " Info"
	if !DllCall("user32\IsWindowVisible", "Ptr",oGuiAcc.Hwnd)
	{
		Border.Hide()
		oSB.SetText("Path: " GetAccPath(Acc).path, 2)
	}
	else
	{
		BuildTreeView()
		oTV.Enabled := True
		WinActivate("ahk_id " oGuiAcc.Hwnd)
		PostMessage(WM_LBUTTONDOWN, 0, 0, "SysTreeView321", "ahk_id " oGuiAcc.Hwnd)
	}
}
#If
~LButton Up::HotkeyLBUp()
HotkeyLBUp()
{
	global
	Hotkey("~LButton Up", "Off")
	LButton_Pressed := False
	if !CH
	{
		oGuiMain["HBar"].Visible := True
		oGuiMain["VBar"].Visible := True
		CrossHair(CH := True)
	}
	Sleep(-1)
	oGuiMain["WinCtrl"].Text := (DllCall("user32\GetParent", "Ptr",Acc_WindowFromObject(Acc), "Ptr") ? "Control":"Window") " Info"
	if !DllCall("user32\IsWindowVisible", "Ptr",oGuiAcc.Hwnd)
	{
		Border.Hide()
		oSB.SetText("Path: " GetAccPath(Acc).path, 2)
	}
	else
	{
		BuildTreeView()
		oTV.Enabled := True
		WinActivate("ahk_id " oGuiAcc.Hwnd)
		PostMessage(WM_LBUTTONDOWN, 0, 0, "SysTreeView321", "ahk_id " oGuiAcc.Hwnd)
	}
}

;functions:
CrossHairSub(oParams*)
{
	global
	if 1
	{
		Hotkey("~LButton Up", "On")
		if 1
		{
			oGuiMain["HBar"].Visible := False
			oGuiMain["VBar"].Visible := False
			CrossHair(CH := False)
		}
		LButton_Pressed := True
		Stored.CHwnd := ""
		oTV.Enabled := False
		while LButton_Pressed
			GetAccInfo()
	}
}
OnExitCleanup(oParams*)
{
	CrossHair(True)
	ExitApp()
}
GuiMainClose(oParams*)
{
	ExitApp()
}
ShowStructureSub(oParams*)
{
	global
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	ControlFocus("Static1", "ahk_id " oGuiMain.Hwnd)
	if DllCall("user32\IsWindowVisible", "Ptr",oGuiAcc.Hwnd)
	{
		PostMessage(WM_LBUTTONDOWN, 0, 0, "SysTreeView321", "ahk_id " oGuiAcc.Hwnd)
		return
	}
	WinGetPos(IsV1?x:&x, IsV1?y:&y, IsV1?w:&w,, "ahk_id " oGuiMain.Hwnd)
	WinGetPos(,, IsV1?AccW:&AccW, IsV1?AccH:&AccH, "ahk_id " oGuiAcc.Hwnd)
	WinMove(x+w+AccW > A_ScreenWidth ? x-AccW-10 : x+w+10, y+5, AccW, AccH, "ahk_id " oGuiAcc.Hwnd)
	WinShow("ahk_id " oGuiAcc.Hwnd)
	if ComObjType(Acc, "Name") = "IAccessible"
		BuildTreeView()
	oTV.Enabled := LButton_Pressed ? False : True
	PostMessage(WM_LBUTTONDOWN, 0, 0, "SysTreeView321", "ahk_id " oGuiAcc.Hwnd)
}
BuildTreeView()
{
	global
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	r := GetAccPath(Acc)
	AccObj := r.AccObj
	, Child_Path := r.Path
	, r := ""
	oTV.Delete()
	oTV.Opt("-Redraw")
	parent := oTV.Add(Acc_Role(AccObj), 0, "Bold Expand")
	TVobj := Map(Parent,{Is_Obj:True, Obj:AccObj, Need_Children:False, ChildID:0, Children:[], Obj_Path:""})
	Loop1 := "."
	Loop1X := IsV1 ? Loop1 : "Loop1"
	Loop Parse, Child_Path, %Loop1X%
	{
		if !IsDigit(A_LoopField)
			TVobj[parent].Obj_Path := Trim(TVobj[parent].Obj_Path "." A_LoopField, ".")
		else
		{
			StoreParent := parent
			parent := TV_BuildAccChildren(AccObj, parent, "", A_LoopField)
			TVobj[parent].Need_Children := False
			TV_Expanded(StoreParent)
			oTV.Modify(parent, "Expand")
			AccObj := TVobj[parent].Obj
		}
	}
	if !ChildID
	{
		TV_BuildAccChildren(AccObj, parent)
		oTV.Modify(parent, "Select")
	}
	else
		TV_BuildAccChildren(AccObj, parent, ChildID)
	TV_Expanded(parent)
	oTV.Opt("+Redraw")
}
GuiAccClose(oParams*)
{
	global
	Border.Hide()
	oGuiAcc.Hide()
	oTV.Delete()
	oGuiMain["ShowStructure"].Enabled := True
}
GuiAccSize(oGui, MinMax, GuiW, GuiH)
{
	global
	oTV.Move(,, GuiW-20, GuiH-12)
}
MainGuiSubmit()
{
	global
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	oCtlContents := oGuiMain.Submit(False) ;False: NoHide
	for VarName, Value in IsV1?oCtlContents:oCtlContents.OwnProps()
	{
		if IsSet(%VarName%)
			%VarName% := Value
	}
}
TreeView_ItemSelect(oCtl, Item)
{
	global
	MainGuiSubmit()
	UpdateAccInfo(TVobj[Item].Obj, TVobj[Item].ChildID, TVobj[Item].Obj_Path)
}
TreeView_ItemExpand(oCtl, Item, Expanded)
{
	global
	MainGuiSubmit()
	oTV.Opt("-Redraw")
	TV_Expanded(Item)
	oTV.Opt("+Redraw")
}
RemoveToolTip()
{
	ToolTip()
}
GetAccInfo()
{
	global ;Acc, ChildID, oGuiMain, Stored, WHwnd, Win
	local class_, Hwnd, Location, parent, posH, posW, posX, posY, proc, procid, SectionLabel, Text, title
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	;static ShowButtonEnabled ;not used by function [FIXME] why present?
	MouseGetPos(,, IsV1?WHwnd:&WHwnd)
	if (WHwnd != oGuiMain.Hwnd) && (WHwnd != oGuiAcc.Hwnd)
	{
		if 1
		{
			SectionLabel := oGuiMain["WinCtrl"].Text
			if (SectionLabel != "Window/Control Info")
				oGuiMain["WinCtrl"].Text := "Window/Control Info"
		}
		Acc := Acc_ObjectFromPoint(IsV1?ChildID:&ChildID)
		Location := GetAccLocation(Acc, IsV1?ChildID:&ChildID)
		if (Stored.Location != Location)
		{
			Hwnd := Acc_WindowFromObject(Acc)
			if (Stored.Hwnd != Hwnd)
			{
				if parent := DllCall("user32\GetParent", "Ptr",Hwnd, "Ptr") ;update: added 'parent' variable [FIXME] correct?
				{
					try
					{
						title := WinGetTitle("ahk_id " parent)
						, text := ControlGetText(Hwnd+0)
						, class_ := ""
						, ControlGetPos(IsV1?posX:&posX, IsV1?posY:&posY, IsV1?posW:&posW, IsV1?posH:&posH, Hwnd+0) ;update: now returns client coordinates (not coordinates relative to window)
						, proc := WinGetProcessName("ahk_id " parent)
						, procid := WinGetPID("ahk_id " parent)
						try class_ := GetClassNN(Hwnd, WHwnd)
					}
					catch
						title := text := class_ := posX := posY := posW := posH := proc := procid := ""
				}
				else
				{
					try
					{
						title := WinGetTitle("ahk_id " Hwnd)
						text := WinGetText("ahk_id " Hwnd)
						class_ := WinGetClass("ahk_id " Hwnd)
						WinGetPos(IsV1?posX:&posX, IsV1?posY:&posY, IsV1?posW:&posW, IsV1?posH:&posH, "ahk_id " Hwnd)
						proc := WinGetProcessName("ahk_id " Hwnd)
						procid := WinGetPID("ahk_id " Hwnd)
					}
					catch
						title := text := class_ := posX := posY := posW := posH := proc := procid := ""
				}
				if 1
				{
					oGuiMain["WinTitle"].Text := title
					oGuiMain["Text"].Text := text
					oGuiMain["Hwnd_"].Text := Format("0x{:X}", Hwnd)
					oGuiMain["Class_"].Text := class_
					oGuiMain["Position"].Text := "x" posX " y" posY
					oGuiMain["Size"].Text := "w" posW " h" posH
					oGuiMain["Process"].Text := proc
					oGuiMain["ProcId"].Text := procid
				}
				Stored.Hwnd := Hwnd
			}
			UpdateAccInfo(Acc, ChildID)
		}
	}
}
UpdateAccInfo(Acc, ChildID, Obj_Path:="")
{
	global ;Border, oGuiMain, oSB, Stored, WHwnd, Win
	local h, Location, w, x, y
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	Location := GetAccLocation(Acc, ChildID, IsV1?x:&x, IsV1?y:&y, IsV1?w:&w, IsV1?h:&h)
	if 1
	{
		oGuiMain["AccName"].Text := ""
		oGuiMain["AccValue"].Text := ""
		oGuiMain["AccRole"].Text := ""
		oGuiMain["AccState"].Text := ""
		oGuiMain["AccAction"].Text := ""
		oGuiMain["AccChildCount"].Text := ""
		oGuiMain["AccSelection"].Text := ""
		oGuiMain["AccFocus"].Text := ""
		oGuiMain["AccLocation"].Text := ""
		oGuiMain["AccDescription"].Text := ""
		oGuiMain["AccKeyboard"].Text := ""
		oGuiMain["AccHelp"].Text := ""
		oGuiMain["AccHelpTopic"].Text := ""

		try oGuiMain["AccName"].Text := Acc.accName[ChildID]
		try oGuiMain["AccValue"].Text := Acc.accValue[ChildID]
		;update: now also shows hex values for role and state:
		;try oGuiMain["AccRole"].Text := Acc_GetRoleText(Acc.accRole[ChildID])
		;try oGuiMain["AccState"].Text := Acc_GetStateText(Acc.accState[ChildID])
		try oGuiMain["AccRole"].Text := Format("0x{:X}", Acc.accRole[ChildID]) " " Acc_GetRoleText(Acc.accRole[ChildID])
		try oGuiMain["AccState"].Text := Format("0x{:X}", Acc.accState[ChildID]) " " Acc_GetStateText(Acc.accState[ChildID])
		try oGuiMain["AccAction"].Text := Acc.accDefaultAction[ChildID]
		try oGuiMain["AccChildCount"].Text := ChildID ? "N/A" : Acc.accChildCount
		try oGuiMain["AccSelection"].Text := ChildID ? "N/A" : Acc.accSelection
		try oGuiMain["AccFocus"].Text := ChildID ? "N/A" : Acc.accFocus ;[FIXME] AHK v1: giving random string value sometimes (e.g. on web browser elements)
		try oGuiMain["AccLocation"].Text := Location
		try oGuiMain["AccDescription"].Text := Acc.accDescription[ChildID]
		try oGuiMain["AccKeyboard"].Text := Acc.accKeyboardShortcut[ChildID]
		try oGuiMain["AccHelp"].Text := Acc.accHelp[ChildID]
		try oGuiMain["AccHelpTopic"].Text := Acc.accHelpTopic[ChildID] ;[FIXME] correct? 1 param or 2?
		oSB.SetText(ChildID ? "Child Id: " ChildID : "Object")
		oSB.SetText(DllCall("user32\IsWindowVisible", "Ptr",oGuiAcc.Hwnd) ? "Path: " Obj_Path : "", 2)
	}
	Border.SetTransparent(True)
	Border.Show(x, y, x+w, y+h)
	Border.SetAbove(WHwnd)
	Border.SetTransparent(False)
	Stored.Location := Location
}
GetClassNN(CHwnd, WHwnd)
{
	global ;_GetClassNN
	local Class_, ClassNN, Detect, EnumAddress
	_GetClassNN := {}
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	_GetClassNN.Hwnd := CHwnd
	Detect := A_DetectHiddenWindows
	Class_ := WinGetClass("ahk_id " CHwnd)
	_GetClassNN.Class := Class_
	_GetClassNN.ClassNN := ""
	DetectHiddenWindows(1)
	EnumAddress := CallbackCreate(IsV1?Func("GetClassNN_EnumChildProc"):GetClassNN_EnumChildProc)
	DllCall("user32\EnumChildWindows", "Ptr",WHwnd, "Ptr",EnumAddress, "Ptr",0)
	DetectHiddenWindows(Detect)
	ClassNN := _GetClassNN.ClassNN
	_GetClassNN := ""
	return ClassNN
}
GetClassNN_EnumChildProc(Hwnd, lParam)
{
	global ;_GetClassNN
	local Class_
	static Occurrence := 0
	try Class_ := WinGetClass("ahk_id " Hwnd)
	catch
		Class_ := ""
	if (_GetClassNN.Class == Class_)
		Occurrence++
	if (_GetClassNN.Hwnd != Hwnd)
		return True
	else
	{
		_GetClassNN.ClassNN := _GetClassNN.Class Occurrence
		Occurrence := 0
		return False
	}
}
TV_Expanded(TVid)
{
	global
	for _, TV_Child_ID in TVobj[TVid].Children
	{
		if TVobj[TV_Child_ID].Need_Children
			TV_BuildAccChildren(TVobj[TV_Child_ID].Obj, TV_Child_ID)
	}
}
TV_BuildAccChildren(AccObj, Parent, Selected_Child:="", Flag:="")
{
	global ;Acc, TVobj
	local _, added, Child, Flagged_Child, Parent_Obj_Path, role
	TVobj[Parent].Need_Children := False
	Parent_Obj_Path := Trim(TVobj[Parent].Obj_Path, ".")
	Flagged_Child := ""
	for _, Child in Acc_ChildrenAlt(AccObj)
	{
		if !IsObject(Child)
		{
			try role := AccObj.accRole[Child]
			catch
				role := 0
			added := oTV.Add("[" A_Index "] " Acc_GetRoleText(role), Parent)
			TVobj[added] := {Is_Obj:False, Obj:Acc, Need_Children:"", ChildID:Child, Children:"", Obj_Path:Parent_Obj_Path}
			if (Child = Selected_Child)
				oTV.Modify(added, "Select")
		}
		else
		{
			added := oTV.Add("[" A_Index "] " Acc_Role(Child), Parent, "bold")
			TVobj[added] := {Is_Obj:True, Obj:Child, Need_Children:True, ChildID:0, Children:[], Obj_Path:Trim(Parent_Obj_Path "." A_Index, ".")}
		}
		TVobj[Parent].Children.Push(added)
		if (A_Index = Flag)
			Flagged_Child := added
	}
	return Flagged_Child
}
GetAccPath(Acc, ByRef Hwnd:="")
{
	local Parent, t1, t2, WinObj, WinObjPos
	Hwnd := Acc_WindowFromObject(Acc)
	WinObj := Acc_ObjectFromWindow(Hwnd, 0) ;note: if use -4, typically get a leading 'P.' in Acc paths
	WinObjPos := Acc_Location(WinObj).pos
	t2 := ""
	while (Acc_WindowFromObject(Parent := Acc_Parent(Acc)) = Hwnd)
	{
		t2 := GetAccEnumIndex(Acc) "." t2
		if (Acc_Location(Parent).pos = WinObjPos)
			return {AccObj:Parent, Path:SubStr(t2, 1, -1)}
		Acc := Parent
	}
	t1 := ""
	while (Acc_WindowFromObject(Parent := Acc_Parent(WinObj)) = Hwnd)
		t1 .= "P."
		, WinObj := Parent
	return {AccObj:Acc, Path:t1 SubStr(t2, 1, -1)}
}
GetAccEnumIndex(Acc, ChildID:=0)
{
	local _, Child, ChildPos
	if !ChildID
	{
		ChildPos := Acc_Location(Acc).pos
		for _, Child in Acc_ChildrenAlt(Acc_Parent(Acc))
		{
			if IsObject(Child) && (Acc_Location(Child).pos = ChildPos)
				return A_Index
		}
	}
	else
	{
		ChildPos := Acc_Location(Acc, ChildID).pos
		for _, Child in Acc_ChildrenAlt(Acc)
		{
			if !IsObject(Child) && (Acc_Location(Acc, Child).pos = ChildPos)
				return A_Index
		}
	}
}
GetAccLocation(AccObj, Child:=0, ByRef x:="", ByRef y:="", ByRef w:="", ByRef h:="")
{
	local RECT
	try
	{
		RECT := Buffer(16, 0)
		AccObj.accLocation(ComValue(0x4003, RECT.Ptr), ComValue(0x4003, RECT.Ptr+4), ComValue(0x4003, RECT.Ptr+8), ComValue(0x4003, RECT.Ptr+12), Child)
		return "x" (x := NumGet(RECT.Ptr, 0, "Int")) "  "
		. "y" (y := NumGet(RECT.Ptr, 4, "Int")) "  "
		. "w" (w := NumGet(RECT.Ptr, 8, "Int")) "  "
		. "h" (h := NumGet(RECT.Ptr, 12, "Int"))
	}
	x := y := w := h := 0
	return "x0  y0  w0  h0"
}
WM_MOUSEMOVE(oParams*)
{
	local ctrl
	static oCur
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	if !IsSet(oCur)
	{
		ClassName := new := "Cursor" ;note: give 'new' a dummy value so AHK v2 doesn't complain
		oCur := IsV1 ? new %ClassName%(32649) : %ClassName%(32649)
	}
	MouseGetPos(,,, IsV1?ctrl:&ctrl)
	if (ctrl = "msctls_statusbar321")
		DllCall("user32\SetCursor", "Ptr",oCur.Handle, "Ptr")
}
CColor(Hwnd, Background:="", Foreground:="")
{
	return CColor_(Background, Foreground, "", Hwnd+0)
}
CColor_(Wp, Lp, Msg, Hwnd)
{
	static
	static WM_CTLCOLOREDIT := 0x0133, WM_CTLCOLORLISTBOX := 0x134, WM_CTLCOLORSTATIC := 0x0138
	, LVM_SETBKCOLOR := 0x1001, LVM_SETTEXTCOLOR := 0x1024, LVM_SETTEXTBKCOLOR := 0x1026, TVM_SETTEXTCOLOR := 0x111E, TVM_SETBKCOLOR := 0x111D
	, BS_CHECKBOX := 2, BS_RADIOBUTTON := 8, ES_READONLY := 0x800
	, CLR_NONE := -1
	, oCol := Map("SILVER",0xC0C0C0, "GRAY",0x808080, "WHITE",0xFFFFFF, "MAROON",0x80, "RED",0x0FF, "PURPLE",0x800080, "FUCHSIA",0xFF00FF, "GREEN",0x8000, "LIME",0xFF00, "OLIVE",0x8080, "YELLOW",0xFFFF, "NAVY",0x800000, "BLUE",0xFF0000, "TEAL",0x808000, "AQUA",0xFFFF00)
	, CLASSES := "Button|ComboBox|Edit|ListBox|Static|RICHEDIT50W|SysListView32|SysTreeView32"
	, CTLCOLORStatic := 0
	, oFG := Map(), oBG := Map(), oBrush := Map()
	, IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	if (Msg = "")
	{
		if !IsSet(adrSetTextColor)
			adrSetTextColor := DllCall("kernel32\GetProcAddress", "Ptr",DllCall("kernel32\GetModuleHandle", "Str","Gdi32.dll", "Ptr"), "AStr","SetTextColor", "Ptr")
			, adrSetBkColor := DllCall("kernel32\GetProcAddress", "Ptr",DllCall("kernel32\GetModuleHandle", "Str","Gdi32.dll", "Ptr"), "AStr","SetBkColor", "Ptr")
			, adrSetBkMode := DllCall("kernel32\GetProcAddress", "Ptr",DllCall("kernel32\GetModuleHandle", "Str","Gdi32.dll", "Ptr"), "AStr","SetBkMode", "Ptr")
		CWp := CLp := ""
		try CWp := oCol[StrUpper(Wp)]
		try CLp := oCol[StrUpper(Lp)]
		BG := !Wp ? "" : CWp != "" ? CWp : "0x" SubStr(Wp, 5, 2) SubStr(Wp, 3, 2) SubStr(Wp, 1, 2)
		FG := !Lp ? "" : CLp != "" ? CLp : "0x" SubStr(Lp, 5, 2) SubStr(Lp, 3, 2) SubStr(Lp, 1, 2)
		class_ := WinGetClass("ahk_id " Hwnd)
		if !RegExMatch(class_, "^(" CLASSES ")$")
			return A_ThisFunc "> Unsupported control class: " class_
		style := ControlGetStyle(Hwnd+0)
		if (class_ = "Edit") && (Style & ES_READONLY)
			class_ := "Static"
		if (class_ = "Button")
		{
			if (style & BS_RADIOBUTTON) || (style & BS_CHECKBOX)
				class_ := "Static"
			else
				return A_ThisFunc "> Unsupported control class: " class_
		}
		if (class_ = "ComboBox")
		{
			Size := (A_PtrSize=8) ? 64 : 52
			CBBINFO := Buffer(Size, 0)
			, _NumPut(Size, CBBINFO.Ptr, 0, "UInt")
			, DllCall("user32\GetComboBoxInfo", "Ptr",Hwnd, "Ptr",CBBINFO.Ptr)
			Hwnd := NumGet(CBBINFO.Ptr, A_PtrSize=8?56:48, "Ptr") ;hwndList
			oBG[Hwnd] := BG
			, oFG[Hwnd] := FG
			, oBrush[Hwnd] := BG ? DllCall("gdi32\CreateSolidBrush", "UInt",BG, "Ptr") : CLR_NONE
			if !CTLCOLORLISTBOX
				CTLCOLORLISTBOX := OnMessage(WM_CTLCOLORLISTBOX, IsV1?A_ThisFunc:%A_ThisFunc%)
			if NumGet(CBBINFO.Ptr, A_PtrSize=8?48:44, "Ptr")
				Hwnd := NumGet(CBBINFO.Ptr, A_PtrSize=8?48:44, "Ptr") ;hwndItem
				, class_ := "Edit"
		}
		if (class_ = "SysListView32") || (class_ = "SysTreeView32")
		{
			m := class_ = "SysListView32" ? "LVM" : "TVM"
			SendMessage(%m%_SETBKCOLOR, 0, BG,, "ahk_id " Hwnd)
			SendMessage(%m%_SETTEXTCOLOR, 0, FG,, "ahk_id " Hwnd)
			SendMessage(%m%_SETTEXTBKCOLOR, 0, CLR_NONE,, "ahk_id " Hwnd)
			return
		}
		if (class_ = "RICHEDIT50W")
		{
			f := "RichEdit_SetBgColor"
			, %f%(Hwnd, -BG)
			return f
		}
		if (!CTLCOLOR%Class_%)
			CTLCOLOR%Class_% := OnMessage(WM_CTLCOLOR%Class_%, IsV1?A_ThisFunc:%A_ThisFunc%)
		oBG[Hwnd] := BG
		, oFG[Hwnd] := FG
		return oBrush[Hwnd] := BG ? DllCall("gdi32\CreateSolidBrush", "UInt",BG, "Ptr") : CLR_NONE
	}
	Critical()
	Hwnd := Lp + 0
	, hDC := Wp + 0
	if (!IsV1 && oBrush.Has(Hwnd))
	|| (IsV1 && oBrush.HasKey(Hwnd))
	{
		DllCall(adrSetBkMode, "Ptr",hDC, "Int",1)
		if oFG[Hwnd]
			DllCall(adrSetTextColor, "Ptr",hDC, "UInt",oFG[Hwnd], "UInt")
		if oBG[Hwnd]
			DllCall(adrSetBkColor, "Ptr",hDC, "UInt",oBG[Hwnd], "UInt")
		return oBrush[Hwnd]
	}
}
CrossHair(OnOff:=1)
{
	local _, system_cursors, Value
	static _s := "", h_cursor, IDC_CROSS := 32515, _c, _b, _h
	if (OnOff = "Init" || OnOff = "I" || _s = "")
	{
		_s := "_h"
		, h_cursor := 0
		, system_cursors := "32512,32513,32514,32515,32516,32642,32643,32644,32645,32646,32648,32649,32650"
		, _c := StrSplit(system_cursors, ",")
		, _b := []
		, _h := []
		for _, Value in _c
		{
			h_cursor := DllCall("user32\LoadCursor", "Ptr",0, "Ptr",Value, "Ptr")
			, _h.Push(DllCall("user32\CopyImage", "Ptr",h_cursor, "UInt",2, "Int",0, "Int",0, "UInt",0, "Ptr"))
			, _b.Push(DllCall("user32\LoadCursor", "Ptr",0, "Ptr",IDC_CROSS, "Ptr"))
		}
	}
	_s := (OnOff = 0 || OnOff = "Off" || _s = "_h" && (OnOff < 0 || OnOff = "Toggle" || OnOff = "T")) ? "_b" : "_h"
	for _, Value in _c
		h_cursor := DllCall("user32\CopyImage", "Ptr",%_s%[A_Index], "UInt",2, "Int",0, "Int",0, "UInt",0, "Ptr")
		, DllCall("user32\SetSystemCursor", "Ptr",h_cursor, "UInt",Value)
}

;note: a version of Acc_Children used by this script only, hence 'Alt' (alternative)
Acc_ChildrenAlt(Acc)
{
	local cChildren, Child, Children, ErrMsg, i, oError, Ret, varChildren
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	if ComObjType(Acc, "Name") != "IAccessible"
		ErrMsg := "Invalid IAccessible Object"
	else
	{
		Acc_Init()
		cChildren := Acc.accChildCount
		Children := []
		if !cChildren
			return Children
		varChildren := Buffer(cChildren*(8+2*A_PtrSize), 0)
		if !DllCall("oleacc\AccessibleChildren", "Ptr",ComObjValue(Acc), "Int",0, "Int",cChildren, "Ptr",varChildren.Ptr, "Int*",IsV1?cChildren:&cChildren)
		{
			;AHK v1: Loop % cChildren
			while (A_Index <= cChildren)
			{
				i := (A_Index-1)*(A_PtrSize*2+8) + 8
				if (NumGet(varChildren.Ptr, i-8, "UShort") == 9) ;child object ;VT_DISPATCH := 9
				{
					Child := NumGet(varChildren.Ptr, i, "UPtr")
					Children.Push(Acc_Query(Child))
					ObjRelease(Child)
				}
				else ;child ID ;VT_I4 := 3
					Children.Push(NumGet(varChildren.Ptr, i, "Int"))
			}
			return Children
		}
		ErrMsg := "AccessibleChildren DllCall Failed"
	}
	oError := Error("", -1)
	Ret := MsgBox((ErrMsg ? ErrMsg "`n`n" : "") "File:`t" (oError.File == A_ScriptFullPath ? A_ScriptName : oError.File) "`nLine:`t" oError.Line "`n`nContinue Script?", "Acc_Children Failed", 0x40004)
	if (Ret = "No")
		ExitApp()
}

_NumPut(vNum, vTarget, vOffset, vType)
{
	static vIsV1
	if !IsSet(vIsV1)
		vIsV1 := (InStr(A_AhkVersion, "1.") == 1)
	if !vIsV1
		return NumPut(vType, vNum, vTarget, vOffset) ;[FOR V2]
	else
		return NumPut(vNum, IsObject(vTarget)?vTarget.Ptr:vTarget+0, vOffset, vType) ;[FOR V1]
}

;Acc Library minus 8 functions (Acc_ObjectFromEvent/Acc_SetWinEventHook/Acc_UnhookWinEvent/Acc_Child/Acc_Error/Acc_Children/Acc_ChildrenByRole/Acc_Get):
Acc_Init()
{
	static h := 0
	if !h
		h := DllCall("kernel32\LoadLibrary", "Str","oleacc", "Ptr")
}

Acc_ObjectFromPoint(ByRef _idChild_:="", x:="", y:="")
{
	local pacc, pt, varChild
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	Acc_Init()
	pt := pacc := 0
	varChild := Buffer(8+2*A_PtrSize, 0)
	if !DllCall("oleacc\AccessibleObjectFromPoint", "UInt64",x == "" || y == "" ? 0*DllCall("user32\GetCursorPos", "Int64*",IsV1?pt:&pt)+pt : x & 0xFFFFFFFF | y << 32, "Ptr*",IsV1?pacc:&pacc, "Ptr",varChild.Ptr)
	{
		_idChild_ := NumGet(varChild.Ptr, 8, "UInt")
		return ComObjFromPtr(pacc)
	}
}

Acc_ObjectFromWindow(hWnd, idObject:=-4)
{
	local IID, pacc
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	Acc_Init()

	static IID_IDispatch := "{00020400-0000-0000-C000-000000000046}"
	static IID_IAccessible := "{618736E0-3C3D-11CF-810C-00AA00389B71}"
	IID := Buffer(16)
	if (idObject & 0xFFFFFFFF = 0xFFFFFFF0) ;OBJID_NATIVEOM := -16 ;(LONG)0xFFFFFFF0
		DllCall("ole32\CLSIDFromString", "WStr",IID_IDispatch, "Ptr",IID.Ptr)
	else
		DllCall("ole32\CLSIDFromString", "WStr",IID_IAccessible, "Ptr",IID.Ptr)

	pacc := 0
	if !DllCall("oleacc\AccessibleObjectFromWindow", "Ptr",hWnd, "UInt",idObject, "Ptr",IID.Ptr, "Ptr*",IsV1?pacc:&pacc)
		return ComObjFromPtr(pacc)
}

Acc_WindowFromObject(pacc)
{
	local hWnd
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	static IID_IAccessible := "{618736E0-3C3D-11CF-810C-00AA00389B71}"
	hWnd := 0
	try
	{
		if !DllCall("oleacc\WindowFromAccessibleObject", "Ptr",IsObject(pacc) ? ComObjValue(pacc) : pacc, "Ptr*",IsV1?hWnd:&hWnd)
			return hWnd
	}
	try
	{
		;appears to be sometimes necessary in AHK v2, but not AHK v1:
		pacc := ComObjQuery(pacc, IID_IAccessible)
		if !DllCall("oleacc\WindowFromAccessibleObject", "Ptr",IsObject(pacc) ? ComObjValue(pacc) : pacc, "Ptr*",IsV1?hWnd:&hWnd)
			return hWnd
	}
	return 0
}

Acc_GetRoleText(nRole)
{
	local nSize, sRole
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	nSize := DllCall("oleacc\GetRoleText", "UInt",nRole, "Ptr",0, "UInt",0, "UInt")
	VarSetStrCapacity(IsV1?sRole:&sRole, nSize)
	DllCall("oleacc\GetRoleText", "UInt",nRole, "Str",sRole, "UInt",nSize+1, "UInt")
	return sRole
}

Acc_GetStateText(nState)
{
	local nSize, sState
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	nSize := DllCall("oleacc\GetStateText", "UInt",nState, "Ptr",0, "UInt",0, "UInt")
	VarSetStrCapacity(IsV1?sState:&sState, nSize)
	DllCall("oleacc\GetStateText", "UInt",nState, "Str",sState, "UInt",nSize+1, "UInt")
	return sState
}

; Written by jethrow
Acc_Role(Acc, ChildID:=0)
{
	try return ComObjType(Acc, "Name") = "IAccessible" ? Acc_GetRoleText(Acc.accRole[ChildID]) : "invalid object"
}
Acc_State(Acc, ChildID:=0)
{
	try return ComObjType(Acc, "Name") = "IAccessible" ? Acc_GetStateText(Acc.accState[ChildID]) : "invalid object"
}
Acc_Location(Acc, ChildID:=0, ByRef Position:="") ; adapted from Sean's code
{
	local pos, RECT
	RECT := Buffer(16, 0)
	try Acc.accLocation(ComValue(0x4003, RECT.Ptr), ComValue(0x4003, RECT.Ptr+4), ComValue(0x4003, RECT.Ptr+8), ComValue(0x4003, RECT.Ptr+12), ChildID)
	catch
	{
		Position := "x  y  w  h "
		return {x:"", y:"", w:"", h:"", pos:Position}
	}
	Position := "x" NumGet(RECT.Ptr, 0, "Int") " y" NumGet(RECT.Ptr, 4, "Int") " w" NumGet(RECT.Ptr, 8, "Int") " h" NumGet(RECT.Ptr, 12, "Int")
	return {x:NumGet(RECT.Ptr, 0, "Int"), y:NumGet(RECT.Ptr, 4, "Int"), w:NumGet(RECT.Ptr, 8, "Int"), h:NumGet(RECT.Ptr, 12, "Int"), pos:Position}
}
Acc_Parent(Acc)
{
	local parent
	try parent := Acc.accParent
	catch
		parent := ""
	return parent ? Acc_Query(parent) : ""
}
Acc_Query(Acc) ; thanks Lexikos - https://www.autohotkey.com/board/topic/76309-closed-help-with-acc-viewer-project/?p=486291
{
	;AHK v1: try return ComObject(9, ComObjQuery(Acc, "{618736e0-3c3d-11cf-810c-00aa00389b71}"), 1) ;IID_IAccessible
	;AHK v1: try return ComValue(9, ComObjQuery(Acc, "{618736e0-3c3d-11cf-810c-00aa00389b71}"), 1) ;IID_IAccessible

	local pAcc
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	try
	{
		pAcc := ComObjQuery(Acc, "{618736e0-3c3d-11cf-810c-00aa00389b71}") ;IID_IAccessible
		ObjAddRef(IsV1?pAcc:pAcc.Ptr)
		return ComValue(9, IsV1?pAcc:pAcc.Ptr) ;VT_DISPATCH := 9
	}
}
